% This script numerically solves the equilibration of a 2 drop system
% in which each drop has an open interface on top and bottom
clear all;

%% SIMULATION SETUP
% ---------------
% Dimensions of the device
% ---------------
Rin   = 2e-3     ; 
Rout  = 2.25e-3  ;
h     = 1.2e-3   ;
wch   = 0.8e-3   ;
hch   = 0.8e-3   ;
lch   = 2e-3     ;
eta   = 1e-3     ;
glam  = (1+wch/hch)^2/(wch^2/hch^2)          ;
Rflu  = 8 * eta * lch * glam / (wch^3 * hch) ;

% ---------------
% Physical constants
% ---------------
global rho g gamma ;
gamma = 0.07       ;
rho   = 1000       ;
g     = 9.8      ;   

% ---------------
% Usesr 
% ---------------
Vmin  = 40e-9    ;
Vmax  = 110e-9   ;

% ---------------
% Calculated volumes
% ---------------
Vc      = lch * wch * hch     ; % volume of the channel
Vin     = h * pi * Rin^2      ; % volume of the port on the input side
Vout    = h * pi * Rout^2     ; % volume of the port on thr output side

% ---------------
% Simulation parameters
% ---------------
simEnd  = 1e-1     ; % Threshold at which the simulation stops
dt      = 0.001      ; % Factor to calculate volume exchange at each step
log     = false     ; % flag for debugging

% ---------------
% Setup arrays to plot the evolution of the radii
% ---------------
n       = 10000           ; % number of points
indeces = linspace(0,n,n) ; % Indexes list

% ---------------
% Get the pressure - volume relationship for the drops
% ---------------

% Make the pressure - volume table for the input drop
PVHS_bot_in  = Function_DropPressureVolume_V03( Rin, 20, -1, false )     ;
display('Determined pressure - volume tables for input pendant drops')  ;

% Make the pressure - volume table for the input drop
PVHS_top_in  = Function_DropPressureVolume_V03( Rin, 20, 1, false )    ;
display('Determined pressure - volume tables for input top drops')      ;

% Make the pressure - volume table for the input drop
PVHS_bot_out = Function_DropPressureVolume_V03( Rout, 20, -1, false )    ;
display('Determined pressure - volume tables for output pendant drops') ;

% Make the pressure - volume table for the input drop
PVHS_top_out = Function_DropPressureVolume_V03( Rout, 20, 1, false )   ;
display('Determined pressure - volume tables for output top drops') ;


%% PRESIMULATION EQUILIBRATION

% ---------------
% Loop through the volumes and calculate the equilibration volumes
% ---------------
V1m  = [] ;
V2m  = [] ;
Vtot = [] ;

for Vol = Vmin : 5e-9 : Vmax
    
    % ---------------
    % Simulation starting variables
    % ---------------
    Ratio   = Vin / (Vout + Vin) ; % Ratio of the volumes of both sides
    V1      = Vol * Ratio         % starting input volume
    V2      = Vol * (1 - Ratio)   % starting output volume

    % ---------------
    % Equilibrate the 2 drops with the initial volume
    % ---------------
    [ ~, ~, V1, V2 ] = Sub_Equilibrate_All_V03( V1, Rin, PVHS_bot_in, PVHS_top_in, V2, Rout, PVHS_bot_out, PVHS_top_out, Rflu, h, simEnd, dt, n, log);
    display('End of droplet equilibration') ;
    
    % ---------------
    % Append the volumes to the vector
    % ---------------
    V1m  = [V1m V1]   ;
    V2m  = [V2m V2]   ;  
    Vtot = [Vtot Vol] ;
    
end


%% PLOTTING

% ---------------
% Plots
% ---------------
figure(5)
plot(Vtot, V1m,'b')                                                   ; 
hold on                                                               ;
plot(Vtot, V2m,'r')                                                   ; 
xlabel('Total volume')                                                ; 
ylabel('Volume in drop')                                              ; 

figure(6)
plot(Vtot, V1m,'b')                                                   ; 
xlabel('Total volume')                                                ; 
ylabel('Volume in drop')                                              ; 

